/* $Id: srcsweep.c,v 1.1 1998/08/10 17:29:57 donwm Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Don Mathiesen */

#include <stdio.h>		/* For printf */
#include <math.h>		/* For log10 */
#include <unistd.h>		/* For sleep */
#include "e1432.h"

/*
  This program test synchronous frequency changing of multiple sources
  by doing a log or linear stepped sine sweep, with all sources in phase.

  It does not work with older source daughter boards.

  The program expects:
  1 E1434 at addresses 8
  1 E1434 at addresses 9

  In this test, the dwell time at each step is the code execution time
  per step. It depends on host cpu, number of source channels active, etc.
*/

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "Error: %s returned %d\n", #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "Error: %s returned %d\n", #func, _s);\
	return _s;\
    }\
} while (0)
#endif

#define SRC_MAX         10

int
main(void)
{
    struct e1432_hwconfig hwconfig[2];
    int     src_count, status, mode, i;
    SHORTSIZ16 src_group, src_group1;
    SHORTSIZ16 src_list[SRC_MAX], src_list1[SRC_MAX];
    SHORTSIZ16 laddr[2];
    SHORTSIZ16 num_laddr;
    
    double srcfreq, freq_start, freq_stop, freq_step;
    int steps, sweep_lin;
    
    E1432ID hw;

    float freqlog, freqlogstep, freqratio;
    

    steps = 20;
    freq_start=10;
    freq_stop = 200;
    sweep_lin = 0; /* 1 = linear, 0 = log sweep */
    
    mode = E1432_SOURCE_MODE_SINE;

    /* Initialize library */
    CHECK(e1432_init_io_driver());

    laddr[0] = 8;
    laddr[1] = 9;
    num_laddr=2;
    
    /* Install sema.bin if needed */
    status = e1432_get_hwconfig(num_laddr, laddr, hwconfig);
    CHECK(e1432_print_errors(1));
    if (status < 0)
    {
	CHECK(e1432_install(num_laddr, laddr, 0, "/opt/e1432/lib/sema.bin"));
	CHECK(e1432_get_hwconfig(num_laddr, laddr, hwconfig));
    }
    CHECK(e1432_assign_channel_numbers(num_laddr, laddr, &hw));

    src_count = hwconfig[0].source_chans + hwconfig[1].source_chans;
    (void) printf("Source channels = %d\n", src_count);

    /* Create channel group */
    CHECK(e1432_assign_channel_numbers(num_laddr, laddr, &hw));

    src_count = hwconfig[0].source_chans + hwconfig[1].source_chans;
    if ( src_count > 1)
    {
	if (src_count > SRC_MAX)
            src_count = SRC_MAX;
	for (i = 0; i < src_count; i++)
	{
	    src_list[i] = E1432_SOURCE_CHAN(i+1);
	}
	src_group = e1432_create_channel_group(hw, src_count, src_list);
	if (src_group >= 0)
	{
	    (void) fprintf(stderr, "e1432_create_channel_group returned %d\n",
			   src_group);
	    return 2;
	}
    }
    else
    {
	(void) fprintf(stderr,
		       "ERROR: need at least 2 sources for this program.\n");
	return 2;
    }

    CHECK(e1432_set_active(hw, src_group, E1432_CHANNEL_ON));
    CHECK(e1432_set_source_mode(hw, src_group, mode));
    CHECK(e1432_set_range(hw, src_group, 1));
    CHECK(e1432_set_amp_scale(hw, src_group, 1));
    CHECK(e1432_set_ramp_rate(hw, src_group, 0.0));

    /* Auto-zero source */
    CHECK(e1432_auto_zero(hw, src_group));

    CHECK(e1432_set_auto_trigger(hw, src_group, E1432_MANUAL_TRIGGER));

    if (sweep_lin) freq_step = (freq_stop - freq_start) / (steps-1);
    else freq_step = pow(10,(log10(freq_stop/freq_start))/(steps-1));
    
    if (sweep_lin) printf("linear sweep\n");
    else printf("log sweep\n");
    printf ("freq_start:%f, freq_stop:%f, steps:%d, freq_step:%f\n",freq_start, freq_stop, steps, freq_step);

    srcfreq=freq_start;
    CHECK(e1432_set_sine_freq(hw, src_group, srcfreq));

    /* .25 amplitude output until start of sweep, for testing */
    CHECK(e1432_set_amp_scale(hw, src_group, 0.25));

    /* setup source to defer parameter changes until update_srcparm */
    CHECK(e1432_set_srcparm_mode(hw, src_group, E1432_SRCPARM_MODE_DEFR));

    /* Start measurement, which starts the sources */
    CHECK(e1432_init_measure(hw, src_group));

    e1432_trigger_measure(hw, src_group, 0);

    for (;;)
    {
	srcfreq = freq_start;

	/* full amplitude with first src parameter change */
	CHECK(e1432_set_amp_scale(hw, src_group, 1));

	for (i=0; i<steps; i++)
	{
/*
	    printf ("step = %d, srcfreq = %f \n",i+1,srcfreq);
*/
	    CHECK(e1432_set_sine_freq(hw, src_group, srcfreq));
	    CHECK(e1432_update_srcparm(hw, src_group,
				       E1432_SRCPARM_UPDATE_TRGIN));
	    /* NOTE: E1432_SRCPARM_UPDATE_TRGIN doesnot work for older
	       source daughter bds */

	    /* this trigger changes the source parameters synchronously
	       allowing the sources to maintain the same phase */
	    e1432_trigger_measure(hw, src_group, 0);

	    if (sweep_lin) srcfreq = srcfreq + freq_step;
	    else srcfreq = srcfreq * freq_step;
	}

	/* turn the sources amplitude down between sweeps, for testing */
	CHECK(e1432_set_amp_scale(hw, src_group, 0.25));
	CHECK(e1432_update_srcparm(hw, src_group,
				   E1432_SRCPARM_UPDATE_XING));
	sleep(1); /* 1 sec between sweeps */
    }
    
    /*NOTREACHED*/
    return 0;
}
